/*
  LCD_4BIT.cpp - Base class that provides LCD Interface With Arduino
 
  LCD_4BIT : Library Interface for Arduino 
  -> Used Direct Pin Arduino Interface LCD
  -> Interface 4 Bit Mode
  -> Support 6 Pin & 7 Pin Control 
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 
 Create : 12 November 2008 by Eakachai Makarn.(ETT CO.,LTD. : WWW.ETT.CO.TH)
*/

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include "wiring.h"
#include "LCD_4BIT.h"

// Function Interface with Used R/W Pin
LCD_4BIT::LCD_4BIT (int rs_pin, int rw_pin, int en_pin,
                    int d4_pin, int d5_pin, int d6_pin, int d7_pin) :
_Used_RW_Pin(1), _rs_pin(rs_pin), _rw_pin(rw_pin), _en_pin(en_pin)

{
  _data_pin[0] = d4_pin;
  _data_pin[1] = d5_pin;
  _data_pin[2] = d6_pin;
  _data_pin[3] = d7_pin; 
}

// Function Interface with Not Used R/W Pin
LCD_4BIT::LCD_4BIT (int rs_pin, int en_pin,
                    int d4_pin, int d5_pin, int d6_pin, int d7_pin) :
_Used_RW_Pin(0), _rs_pin(rs_pin), _en_pin(en_pin)

{
  _data_pin[0] = d4_pin;
  _data_pin[1] = d5_pin;
  _data_pin[2] = d6_pin;
  _data_pin[3] = d7_pin; 
}

// Initial Character LCD Display
void LCD_4BIT::Initial(void) 
{
  pinMode(_rs_pin,OUTPUT);				// Initial Signal Interface  
  pinMode(_en_pin,OUTPUT);

  if (_Used_RW_Pin) 
  {
    pinMode(_rw_pin,OUTPUT);
  }

  for (int i = 0; i < 4; i++)
  {
    pinMode(_data_pin[i], OUTPUT);
  }

  digitalWrite(_rs_pin,LOW); 				// Standby RS
  
  digitalWrite(_en_pin,LOW); 				// Standby EN

  if (_Used_RW_Pin) 
  {
    digitalWrite(_rw_pin,LOW); 		
  }

  delay(50);						// Power-On Delay
  
  // Start of 4 Bit LCD Interface Initial
  _Write_LCD_Nibble(0x30, true);			// Step[1] = D7:D6:D4:D5 = 0:0:1:1
  delay(5);						      // Wait 4.1mS
  _Write_LCD_Nibble(0x30, true);  			// Step[2] = D7:D6:D4:D5 = 0:0:1:1
  delayMicroseconds(100);				// Wait 100uS
  _Write_LCD_Nibble(0x30, true);  			// Step[3] = D7:D6:D4:D5 = 0:0:1:1
  _Busy_LCD(true);					// Wait Busy Command
  _Write_LCD_Nibble(0x20, true);  			// Step[4] = D7:D6:D4:D5 = 0:0:1:0
  _Busy_LCD(true);					// Wait Busy Command
  // End of 4 Bit LCD Interface Initial

  Command(0x28);  					// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)
  Command(0x0C);  					// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  Command(0x06);  					// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)
  Command(0x01);  					// Clear Display  (Clear Display,Set DD RAM Address=0
}

// Write Command to LCD
void LCD_4BIT::Command(int value) 
{
  _Write_LCD_Byte(value, true);			// Command
  _Busy_LCD(true);					// Wait Busy Command
}

// Print Value of Unsigned Char
void LCD_4BIT::Print(uint8_t b)
{
  _Write_LCD_Byte(b, false);				// Write Data to LCD
  _Busy_LCD(false);					// Wait Busy Data
}

//Print Value of Character
void LCD_4BIT::Print(char c)
{
  Print((byte) c);
}

// Print Value of Constant
void LCD_4BIT::Print(const char c[])
{
  while (*c)
  {
    Print(*c++);
  }
}

// Print Value of Singed Integer
void LCD_4BIT::Print(int n)
{
  Print((long) n);
}

// Print Value of Unsigned Integer
void LCD_4BIT::Print(unsigned int n)
{
  Print((unsigned long) n);
}

// Print Value of Long
void LCD_4BIT::Print(long n)
{
  if (n < 0) 
  {
    Print('-');
    n = -n;
  }
  _printNumber(n, 10);
}

// Print Value of Unsigned Long
void LCD_4BIT::Print(unsigned long n)
{
  _printNumber(n, 10);
}

// Print Value of Long with Format(BYTE,BCD,BIN,OCT,HEX,DEC)
void LCD_4BIT::Print(long n, int base)
{
  if (base == BYTE)					//BYTE
  {
    Print((char) n);
  }
  else if (base == BCD)					//BCD
  {
    if (n < 0x10)
    {
      Print("0");
    }  
    Print(n, HEX);
  }
  else if (base == DEC)					//DEC
  {
    Print(n);
  }
  else							//BIN,OCT,HEX
  {
    _printNumber(n, base);
  }
}

// Set Cursor Position
void LCD_4BIT::SetCursor(int Cursor)
{
  Command(Cursor | 0x80);				// Set DDRAM Command = D7 Set
}

// Clear Screen Display
void LCD_4BIT::ClearScreen()
{
  Command(lcd_clear_screen);				// Clear Screen Display
}

// Clear LCD From Cursor with Length
void LCD_4BIT::ClearLCD(int Cursor, int length)
{
  SetCursor(Cursor);					// Set Cursor for Start Clear Display
  for(int i=0; i<length; i++)
  {
    Print(0x20,BYTE);  					// Write Space for Clear Display
  }
  SetCursor(Cursor);					// Set Cursor For Start New Display
}

// Read Display(ASCII) From LCD Display
int LCD_4BIT::ReadDisplay(void)
{ 
  int LCD_Data = 0;
  
  if (_Used_RW_Pin) 
  {
    LCD_Data = _ReadLCD(false);			// Read LCD Display(Data)
  } 
  else
  {
    LCD_Data = '?';
  }

  return LCD_Data;
}

// Read Command & Address From LCD Display
int LCD_4BIT::ReadAddress(void)
{ 
  int LCD_Address = 0;
  
  if (_Used_RW_Pin) 
  {
    LCD_Address = (_ReadLCD(true)&0x7F);		// Read LCD Address(Command)
  } 
  else
  {
    LCD_Address = 0x00;
  }

  return LCD_Address;
}

// Read Busy LCD Status
void LCD_4BIT::_Busy_LCD(bool lcd_command)
{ 
  int val = 0;

  if (_Used_RW_Pin) 
  {   
    //Loop Read & Wait Busy = 0(0=Ready, 1=Busy)
    do
    {      
      val = (_ReadLCD(true)&0x80);			// Read LCD Command & Verify Bit D7 Only      
    }
    while(val==0x80);					// Wait Busy = 1(Reday)
  } 
  else
  {
    if (!lcd_command) 
    {
      delayMicroseconds(100);				// Delay Data = 100uS
    }
    else
    { 
      delay(2);						// Delay Command = 2mS
    }
  }
}

// Read ASCII From LCD Display
int LCD_4BIT::_ReadLCD(bool lcd_command)		// False=Data, True=Command
{ 
  int LCD_Data = 0;
  
  if (_Used_RW_Pin) 
  {
    digitalWrite(_rw_pin, HIGH);			// RW = HIGH(Read)

    if (!lcd_command) 
    {
      digitalWrite(_rs_pin, HIGH);			// RS = HIGH(Data)    
    }
    else
    {
      digitalWrite(_rs_pin, LOW);			// RS = LOW(Command)     
    }
    
    for (int i = 0; i < 4; i++)			// Config LCD Data = Input
    {
      pinMode(_data_pin[i], INPUT);
    }
    delayMicroseconds(1);				// Delay Pulse = 1uS

    //Read MSB Data
    digitalWrite(_en_pin, HIGH);			// Enable = HIGH(Start Enable)
    delayMicroseconds(1);				// Delay Pulse = 1uS
    for (int i = 0; i < 4; i++) 
    {
      LCD_Data |= (digitalRead(_data_pin[i])<<(i+4));
    }
    digitalWrite(_en_pin, LOW);			// Enable = LOW(Standby)
    delayMicroseconds(1);				// Delay Pulse = 1uS

    //Read LSB Data
    digitalWrite(_en_pin, HIGH);			// Enable = HIGH(Start Enable)
    delayMicroseconds(1);				// Delay Pulse = 1uS
    for (int i = 0; i < 4; i++) 
    {
      LCD_Data |= (digitalRead(_data_pin[i])<<i);
    }
    digitalWrite(_en_pin, LOW);			// Enable = LOW(Standby)
    delayMicroseconds(1);				// Delay Pulse = 1uS
        
    for (int i = 0; i < 4; i++)			// Config LCD Data = Output
    {
      pinMode(_data_pin[i], OUTPUT);
    }
    digitalWrite(_rw_pin, LOW);			// RW = LOW(Write)
    digitalWrite(_en_pin, LOW);			// Enable = LOW(Standby)
  } 
  else
  {
    LCD_Data = '?';
  }

  return LCD_Data;
}

// Write Byte to LCD
void LCD_4BIT::_Write_LCD_Byte(int value, bool lcd_command) 
{
  int nibble = 0;

  nibble = value & 0xF0;     				// Send LCD Data MSB 4 Bit
  _Write_LCD_Nibble(nibble, lcd_command);
 
  nibble = (value << 4) & 0xF0; 			// Send LCD Data LSB 4 Bit 
  _Write_LCD_Nibble(nibble, lcd_command);  
}

// Send 4 Bit Data to LCD
void LCD_4BIT::_Write_LCD_Nibble(int nibble, bool lcd_command) 
{  
  if (!lcd_command) 
  {
    digitalWrite(_rs_pin, HIGH);			// RS = HIGH(Data)    
  }
  else
  {
    digitalWrite(_rs_pin, LOW);			// RS = LOW(Command)    
  }

  if (_Used_RW_Pin) 
  {
    digitalWrite(_rw_pin, LOW);			// RW = LOW(Write)
  }

  for (int i = 0; i < 4; i++) 
  {
    digitalWrite(_data_pin[i], (nibble >> (i + 4)) & 0x01);
  }
 
  digitalWrite(_en_pin, HIGH);			// Enable = HIGH(Start Enable)
  delayMicroseconds(1);					// Delay Pulse = 1uS
  digitalWrite(_en_pin, LOW);				// Enable = LOW(Standby)
}

void LCD_4BIT::_printNumber(unsigned long n, uint8_t base)
{
  unsigned char buf[8 * sizeof(long)]; 		// Assumes 8-bit chars. 
  unsigned long i = 0;

  if (n == 0) 						
  {
    Print('0');
    return;
  } 

  while (n > 0) 
  {
    buf[i++] = n % base;
    n /= base;
  }

  for (; i > 0; i--)
  {
    Print((char) (buf[i - 1] < 10 ?
                  '0' + buf[i - 1] :		// 0..9
                  'A' + buf[i - 1] - 10));	// A..F
  }
}
